// Copyright Nuno Afonso

#pragma once

#include "FArkSourceControlCommand.h"
#include "ISourceControlProvider.h"
#include "Ark_Plugin.h"
#include "Widgets/Notifications/SNotificationList.h"

class SWidget;
class FArkSourceControlState;

struct FArkSourceControlRequest {
	uint64_t request_id = 0;
	uint64_t command_id = 0;
};

struct FArkSourceControlNotification {
	uint64_t notification_id = 0;
	TSharedPtr<SNotificationItem> notification;
};

struct FArkNotificationInfo {
	FNotificationInfo base_info;
	uint64_t notification_id;
};

/* Allows you to use Ark Version Control System from within Unreal Editor.
 * Code convention may ruffle some feathers, but basically it is following 
 * a mix of Unreal and Ark's, sorry if you are looking through it and find 
 * it unpleasant.
 */
class FArkSourceControlProvider : public ISourceControlProvider
{
public:

	FArkSourceControlProvider();
	virtual ~FArkSourceControlProvider() override = default;

	const FString& get_workspace_full_path();

	FString get_relative_path(const FString& absolute_path);
	FString get_absolute_path(const FString& relative_path);

	TArray<FString> get_relative_paths(const TArray<FString>& files);

	bool is_initialized(const FString& path);
	bool initialize_workspace(bool only_update_host_and_email = false);

	bool launch_ark();
	bool connect_to_ark(uint32 process_id);

	void process_command_sync(FArkSourceControlCommand* command);

	void push_request(FArkSourceControlCommand* command, uint64_t request_id);

	FArkSourceControlState* get_file_state(const FString& relative_path, bool* out_was_added = nullptr);
	FSourceControlStateRef* get_file_state_ref(const FString& relative_path, bool* out_was_added = nullptr);

	FSourceControlChangelistStateRef get_ws_cl_or_add(int64 ws_cl_id);
	void clear_all_ws_cls();
	void delete_ws_cl(int64_t ws_cl_id);
	void update_ws_cls(bool async = true);

	void force_update_relative_paths(const TArray<FString>& relative_paths);
	void force_update_current_directory_relative_paths();
	
	bool get_asset(const FString& relative_path, uint32 cl_id, uint32 cl_revision, FString& out_absolute_path, EConcurrency::Type concurrency);


	void* ark_library_handle = nullptr;
	Ark_Plugin* plugin = nullptr;

	bool is_ark_running = false;
	FProcHandle ark_proc_handle;
	
	TArray<FArkSourceControlCommand*> commands;
	TArray<FArkSourceControlRequest> requests;

	TMap<FString, FSourceControlStateRef> relative_path_to_file_state_ref;
	TMap<int64, FSourceControlChangelistStateRef> ws_cls;
	
	FText status_text;

	void show_notification_request_lock_ownership(const FString& relative_path, const FString& lock_owner);
	FArkNotificationInfo create_notification_info(const FString& text, const FString& sub_text);
	void set_notification_state(uint64_t notification_id, SNotificationItem::ECompletionState state);
	TArray<FArkSourceControlNotification> notifications;
	// Since we also receive notifications with ids from ark, we need an offset for out ue specific notifications
	uint64_t last_ue_notification_id = 0x700000000;

	FName provider_name = FName("Ark Vcs");
	FSourceControlStateChanged on_source_control_state_changed;

	static void* plugin_alloc       (int64_t size);
	static void  plugin_free        (void* memory);
	static void* plugin_realloc     (void* old_memory, int64_t size, int64_t old_size);
	static void  plugin_logger      (Ark_Plugin_String message, uint8_t log_level);
	static void  plugin_response    (Ark_Plugin_Response *response);
	static void  plugin_notification(Ark_Plugin_Notification *notification);

	// ISourceControlProvider

	virtual void Init(bool bForceConnection = true) override;
	virtual void Close() override;
	virtual const FName& GetName() const override { return provider_name; }
	virtual FText GetStatusText() const override;
	virtual TMap<EStatus, FString> GetStatus() const override;
	virtual bool IsEnabled() const override;
	virtual bool IsAvailable() const override;

	// No clue how these are meant to work
	virtual bool QueryStateBranchConfig(const FString& ConfigSrc, const FString& ConfigDest) override { return false; }
	virtual void RegisterStateBranches(const TArray<FString>& BranchNames, const FString& ContentRoot) override {}
	virtual int32 GetStateBranchIndex(const FString& BranchName) const override { return -1; }
#if ENGINE_MAJOR_VERSION >=5 && ENGINE_MINOR_VERSION >= 7
	virtual bool GetStateBranchAtIndex(int32 BranchIndex, FString& OutBranchName) const { return false; }
#endif // ENGINE_MAJOR_VERSION >=5 && ENGINE_MINOR_VERSION >= 7

	virtual ECommandResult::Type GetState(const TArray<FString>& InFiles, TArray<FSourceControlStateRef>& OutState, EStateCacheUsage::Type InStateCacheUsage) override;
	virtual ECommandResult::Type GetState(const TArray<FSourceControlChangelistRef>& InChangelists, TArray<FSourceControlChangelistStateRef>& OutState, EStateCacheUsage::Type InStateCacheUsage) override;
	virtual TArray<FSourceControlStateRef> GetCachedStateByPredicate(TFunctionRef<bool(const FSourceControlStateRef&)> Predicate) const override;
	virtual FDelegateHandle RegisterSourceControlStateChanged_Handle(const FSourceControlStateChanged::FDelegate& SourceControlStateChanged) override;
	virtual void UnregisterSourceControlStateChanged_Handle(FDelegateHandle Handle) override;
	virtual ECommandResult::Type Execute(const FSourceControlOperationRef& InOperation,	FSourceControlChangelistPtr InChangelist, const TArray<FString>& InFiles, EConcurrency::Type InConcurrency,	const FSourceControlOperationComplete& InOperationCompleteDelegate) override;
	virtual bool CanExecuteOperation(const FSourceControlOperationRef& InOperation) const override;
	virtual bool CanCancelOperation(const FSourceControlOperationRef& InOperation) const override;
	virtual void CancelOperation(const FSourceControlOperationRef& InOperation) override;
	virtual TArray<TSharedRef<ISourceControlLabel>> GetLabels(const FString& InMatchingSpec) const override;
	virtual TArray<FSourceControlChangelistRef> GetChangelists(EStateCacheUsage::Type InStateCacheUsage) override;
	
	virtual bool UsesLocalReadOnlyState() const override { return false; }
	virtual bool UsesChangelists() const override { return true; }
	virtual bool UsesUncontrolledChangelists() const override { return false; }
	virtual bool UsesCheckout() const override { return true; }
	virtual bool UsesFileRevisions() const override { return true; }
	virtual bool UsesSnapshots() const override { return false; }
	virtual bool AllowsDiffAgainstDepot() const override { return true; }
	virtual TOptional<bool> IsAtLatestRevision() const override;
	virtual TOptional<int> GetNumLocalChanges() const override;
	virtual void Tick() override;
#if SOURCE_CONTROL_WITH_SLATE
	virtual TSharedRef<SWidget> MakeSettingsWidget() const override;
#endif // SOURCE_CONTROL_WITH_SLATE
};
